import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";

type TaskAnalysisData = {
  taskStats: {
    totalTasks: number;
    completedTasks: number;
    pendingTasks: number;
    overdueTasks: number;
    recurringTasks: number;
    nonRecurringTasks: number;
    periodCompletionRate: number;
    totalPeriods: number;
    completedPeriods: number;
    tasksCreatedToday: number;
    tasksCreatedThisWeek: number;
    tasksCreatedThisMonth: number;
  };
  patternStats: Array<{
    pattern: string;
    total: number;
    completed: number;
    completionRate: number;
  }>;
  departmentStats: Array<{
    department: string;
    total: number;
    completed: number;
    overdue: number;
    nonConformities: number;
    completionRate: number;
  }>;
  technicianStats: Array<{
    technicianId: string;
    technicianName: string;
    technicianEmail: string;
    isExternal: boolean;
    companyName: string | null;
    total: number;
    completed: number;
    overdue: number;
    avgCompletionTime: number;
    completionRate: number;
  }>;
  nonConformityStats: {
    totalNonConformities: number;
    tasksWithNonConformities: number;
    nonConformityRate: number;
  };
};

interface TaskOverviewTabProps {
  taskData: TaskAnalysisData;
}

export default function TaskOverviewTab({ taskData }: TaskOverviewTabProps) {
  const { t } = useLanguage();
  const { taskStats, patternStats, nonConformityStats } = taskData;
  const completionRate =
    taskStats.totalTasks > 0
      ? (taskStats.completedTasks / taskStats.totalTasks) * 100
      : 0;

  return (
    <>
      {/* Main Statistics */}
      <div className='grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.totalTasks") || "Total Tasks"}
          </h3>
          <p className='text-2xl font-bold text-gray-900'>
            {taskStats.totalTasks}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.completed") || "Completed"}
          </h3>
          <p className='text-2xl font-bold text-green-600'>
            {taskStats.completedTasks}
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {completionRate.toFixed(1)}% {t("analysis.taskAnalysis.completionRate") || "completion rate"}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.pending") || "Pending"}
          </h3>
          <p className='text-2xl font-bold text-orange-600'>
            {taskStats.pendingTasks}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.overdue") || "Overdue"}
          </h3>
          <p className='text-2xl font-bold text-red-600'>
            {taskStats.overdueTasks}
          </p>
        </div>
      </div>

      {/* Task Type Statistics */}
      <div className='grid grid-cols-1 md:grid-cols-2 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-lg font-semibold mb-4'>
            {t("analysis.taskAnalysis.taskTypes") || "Task Types"}
          </h3>
          <div className='space-y-3'>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.taskAnalysis.recurringTasks") || "Recurring Tasks"}
              </span>
              <div className='flex items-center gap-2'>
                <span className='text-lg font-bold text-gray-900'>
                  {taskStats.recurringTasks}
                </span>
                <span className='text-xs text-gray-500'>
                  ({taskStats.periodCompletionRate.toFixed(1)}% {t("analysis.taskAnalysis.periodCompletion") || "period completion"})
                </span>
              </div>
            </div>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.taskAnalysis.nonRecurringTasks") || "Non-Recurring Tasks"}
              </span>
              <span className='text-lg font-bold text-gray-900'>
                {taskStats.nonRecurringTasks}
              </span>
            </div>
            <div className='pt-3 border-t'>
              <div className='flex justify-between items-center'>
                <span className='text-sm font-medium text-gray-700'>
                  {t("analysis.taskAnalysis.totalPeriods") || "Total Periods"}
                </span>
                <span className='text-sm text-gray-600'>
                  {taskStats.completedPeriods} / {taskStats.totalPeriods} {t("analysis.taskAnalysis.completedPeriods") || "completed"}
                </span>
              </div>
            </div>
          </div>
        </div>

        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-lg font-semibold mb-4'>
            {t("analysis.taskAnalysis.nonConformities") || "Non-Conformities"}
          </h3>
          <div className='space-y-3'>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.taskAnalysis.totalIssues") || "Total Issues"}
              </span>
              <span className='text-lg font-bold text-red-600'>
                {nonConformityStats.totalNonConformities}
              </span>
            </div>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.taskAnalysis.tasksWithIssues") || "Tasks with Issues"}
              </span>
              <span className='text-lg font-bold text-orange-600'>
                {nonConformityStats.tasksWithNonConformities}
              </span>
            </div>
            <div className='pt-3 border-t'>
              <div className='flex justify-between items-center'>
                <span className='text-sm font-medium text-gray-700'>
                  {t("analysis.taskAnalysis.nonConformityRate") || "Non-Conformity Rate"}
                </span>
                <span className='text-sm text-gray-600'>
                  {nonConformityStats.nonConformityRate.toFixed(1)}%
                </span>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Recurrence Pattern Statistics */}
      <div className='bg-white p-6 rounded-lg shadow mb-8'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.completionByPattern") || "Completion by Recurrence Pattern"}
        </h3>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.pattern") || "Pattern"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.total") || "Total"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completed") || "Completed"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {patternStats.map((pattern) => (
                <tr key={pattern.pattern}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 capitalize'>
                    {pattern.pattern}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {pattern.total}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {pattern.completed}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm'>
                    <span
                      className={`font-medium ${
                        pattern.completionRate >= 80
                          ? "text-green-600"
                          : pattern.completionRate >= 50
                          ? "text-orange-600"
                          : "text-red-600"
                      }`}
                    >
                      {pattern.completionRate.toFixed(1)}%
                    </span>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Time-based Statistics */}
      <div className='grid grid-cols-1 md:grid-cols-3 gap-6'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.createdToday") || "Created Today"}
          </h3>
          <p className='text-2xl font-bold text-gray-900'>
            {taskStats.tasksCreatedToday}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.createdThisWeek") || "Created This Week"}
          </h3>
          <p className='text-2xl font-bold text-gray-900'>
            {taskStats.tasksCreatedThisWeek}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500'>
            {t("analysis.taskAnalysis.createdThisMonth") || "Created This Month"}
          </h3>
          <p className='text-2xl font-bold text-gray-900'>
            {taskStats.tasksCreatedThisMonth}
          </p>
        </div>
      </div>
    </>
  );
}

